/* Copyright (C) 2012-2018 RealVNC Ltd. All Rights Reserved.
 *
 * This is sample code intended to demonstrate part of the
 * VNC Mobile Solution SDK. It is not intended as a production-ready
 * component.
 */

struct libusb_device_handle;
struct libusb_device;
struct libusb_context;
struct libusb_interface_descriptor;

#include "UsbDeviceProvider.h"
#include <usbdeviceproviderdefs.h>
#ifdef _WIN32_WCE
#include <Winsock2.h>
#endif
#ifndef VNC_NO_LIBUSB
#include <libusb.h>
#endif // VNC_NO_LIBUSB
#ifndef _WIN32
#include <unistd.h>
#endif
#include <MemHelpers.h>
#include <memory>

#define BUS_DEV_NUM_SIZE 3
#define VID_PID_SIZE 4
#define USB_CLASS_SIZE 3
#define MAX_SERIAL_SIZE 128
#define CONFIG_VALUE_SIZE 3
#define MAX_DEVPATH 128

#define LIBUSB_ERROR(msg, err) LOG_ERROR_F(this, msg \
        ". Libusb error %d.", err);
#define LIBUSB_WARNING(msg, err) LOG_WARNING_F(this, msg \
        ". Libusb error %d.", err);

// Interface classes that can be of interest to discovery. Any other interface
// classes will be ignored.
#define CDC_CONTROL_IF_CLASS 0x02
#define HID_IF_CLASS 0x03
#define STILL_IMAGE_IF_CLASS 0x06
#define MASS_STORAGE_IF_CLASS 0x08
#define CDC_DATA_IF_CLASS 0x0A
#define VIDEO_IF_CLASS 0x0E
#define AUDIO_VIDEO_DEVICE_IF_CLASS 0x10
#define WIRELESS_CONTROLLER_IF_CLASS 0xE0
#define ACTIVE_SYNC_IF_CLASS 0xEF
#define APPLICATION_SPECIFIC_IF_CLASS 0xFE
#define VENDOR_SPECIFIC_IF_CLASS 0xFF


#define HID_KB_SUBCLASS 0x01
#define HID_KB_PROTOCOL 0x01
#define HID_MOUSE_SUBCLASS 0x01
#define HID_MOUSE_PROTOCOL 0x02

#define ABSTRACT_MODEM_SUBCLASS 0x02
#define MASS_STORAGE_SCSI_IF_SUBCLASS 0x06

// Descriptor type for interface association descriptors
#define INTERFACE_ASSOCIATION_TYPE 11
#define INTERFACE_ASSOCIATION_MIN_LENGTH 8

#define CDC_CONTROL_IF_CLASS_STR "2"
#define CDC_CONTROL_NCM_SUBCLASS_STR "13"

namespace
{
  /**
   * Holds an array of unsigned char* pointers.
   * This class has ownership of the unsigned char* so
   * will delete them on destruction using delete[].
   */
  class ConfigDescriptors
  {
  public:
    ConfigDescriptors()
      : mConfigs(NULL)
      , mConfigsSize(0)
      , mConfigLengths(NULL)
    {
    }
    ~ConfigDescriptors()
    {
      destroy();
    }
    bool allocate(size_t newSize)
    {
      destroy();
      mConfigs = new (std::nothrow) unsigned char*[newSize];
      if (!mConfigs)
      {
        return false;
      }
      mConfigLengths = new (std::nothrow) size_t[newSize];
      if (!mConfigLengths)
      {
        destroy();
        return false;
      }
      mConfigsSize = newSize;
      for (size_t i = 0; i < mConfigsSize; ++i)
      {
        mConfigs[i] = NULL;
        mConfigLengths[i] = 0;
      }
      return true;
    }
    bool allocItem(size_t idx, size_t itemSize)
    {
      freeItem(idx);
      mConfigs[idx] = new (std::nothrow) unsigned char[itemSize];
      if (mConfigs[idx])
      {
        mConfigLengths[idx] = itemSize;
        return true;
      }
      return false;
    }
    void freeItem(size_t idx)
    {
      delete[] mConfigs[idx];
      mConfigs[idx] = NULL;
      mConfigLengths[idx] = 0;
    }
    unsigned char *& operator[](size_t idx)
    {
      assert(mConfigs);
      assert(idx < mConfigsSize);
      return mConfigs[idx];
    }
    size_t& length(size_t idx)
    {
      assert(mConfigLengths);
      assert(idx < mConfigsSize);
      return mConfigLengths[idx];
    }
    size_t size() const
    {
      return mConfigsSize;
    }
  private:
    void destroy()
    {
      for (size_t i = 0; i < mConfigsSize; ++i)
      {
        freeItem(i);
      }
      delete[] mConfigLengths;
      delete[] mConfigs;
      mConfigs = NULL;
      mConfigsSize = 0;
      mConfigLengths = NULL;
    }
  private:
    unsigned char** mConfigs;
    size_t mConfigsSize;
    size_t * mConfigLengths;
  };

  int getConfigDescriptors(libusb_device_handle* pDevHandle,
                           ConfigDescriptors& descs)
  {
#ifndef VNC_NO_LIBUSB
    for (size_t i = 0; i < descs.size(); ++i)
    {
      /* A buffer just big enough to hold the config descriptor size */
      unsigned char confBuf[LIBUSB_DT_CONFIG_SIZE];
      int ret = libusb_get_descriptor(pDevHandle,
          LIBUSB_DT_CONFIG, i,
          confBuf, sizeof(confBuf));
      if (ret < 0)
      {
        return ret;
      }
      if (ret < LIBUSB_DT_CONFIG_SIZE ||
          confBuf[0] != LIBUSB_DT_CONFIG_SIZE ||
          confBuf[1] != LIBUSB_DT_CONFIG)
      {
        // Short transfer or invalid descriptor
        return LIBUSB_ERROR_OTHER;
      }
      // Read little-endian value
      const ssize_t wTotalLen = confBuf[2] | confBuf[3] << 8;
      if (!descs.allocItem(i, wTotalLen))
      {
        return LIBUSB_ERROR_NO_MEM;
      }
      ret = libusb_get_descriptor(pDevHandle,
          LIBUSB_DT_CONFIG, i,
          descs[i], wTotalLen);
      if (ret < 0)
      {
        descs.freeItem(i);
        return ret;
      }
      if (ret < wTotalLen)
      {
        descs.freeItem(i);
        return LIBUSB_ERROR_OTHER;
      }
    }
    return LIBUSB_SUCCESS;
#else
    return 0;
#endif // VNC_NO_LIBUSB
  }

  /* Reads the next descriptor, assigning descriptor to
   * the start of the descriptor and updating bLength
   * and bDescriptorType if this returns true.
   * start is always updated to point at the next place
   * to consider for a descriptor. */
  bool getNextDescriptor(const unsigned char*& start,
      const unsigned char* end,
      const unsigned char*& descriptor,
      vnc_uint8_t& bLength, vnc_uint8_t& bDescriptorType)
  {
    while(start + 1 < end)
    {
      descriptor = start;
      bLength = descriptor[0];
      bDescriptorType = descriptor[1];
      if (bLength < 2 ||
          start + bLength >= end)
      {
        // Descriptor goes off the end or is too short, give up
        start = end;
        return false;
      }
      else
      {
        // Found a valid descriptor
        start += bLength;
        return true;
      }
    }
    // Failed to find any valid descriptor
    start = end;
    return false;
  }

  size_t countInterfaceAssociations(const unsigned char* start,
      const unsigned char* end)
  {
    size_t count = 0;
    const unsigned char* desc = NULL;
    vnc_uint8_t bLength = 0;
    vnc_uint8_t bDescriptorType = 0;
    while (getNextDescriptor(start, end, desc, bLength, bDescriptorType))
    {
      if (bDescriptorType == INTERFACE_ASSOCIATION_TYPE &&
          bLength >= INTERFACE_ASSOCIATION_MIN_LENGTH)
      {
        ++count;
      }
    }
    return count;
  }

  /**
   * Retrieves the device which has the bus number and device number specified.
   *
   * \return The usb device that has the given bus and device numbers. NULL if
   * the device can't be found. libusb_unref_device must be called to release
   * the device when no longer needed.
   */
  static libusb_device* getUsbDevice(libusb_context* pLibusbCtx,
                                vnc_uint8_t busNum,
                                vnc_uint8_t devNum)
  {
#ifndef VNC_NO_LIBUSB
    libusb_device** ppDeviceList = NULL;
    libusb_device* ppDevice = NULL;
    const ssize_t deviceCount = libusb_get_device_list(pLibusbCtx, &ppDeviceList);

    for(int i = 0; i < deviceCount; ++i)
    {
      if(libusb_get_bus_number(ppDeviceList[i]) == busNum &&
         libusb_get_device_address(ppDeviceList[i]) == devNum)
      {
        // Found the desired device
        ppDevice = libusb_ref_device(ppDeviceList[i]);
        break;
      }
    }
    libusb_free_device_list(ppDeviceList, 1);

    return ppDevice;
#else
    return NULL;
#endif // VNC_NO_LIBUSB
  }

  static bool isInterfaceOfInterest(
      const libusb_interface_descriptor* pIfrDescriptor)
  {
#ifndef VNC_NO_LIBUSB

    // Allow through interfaces that are likely to be present on a device
    // capable of running a VNC server.
    if(pIfrDescriptor->bInterfaceClass == CDC_CONTROL_IF_CLASS ||
        pIfrDescriptor->bInterfaceClass == STILL_IMAGE_IF_CLASS ||
        pIfrDescriptor->bInterfaceClass == CDC_DATA_IF_CLASS ||
        pIfrDescriptor->bInterfaceClass == VIDEO_IF_CLASS ||
        pIfrDescriptor->bInterfaceClass == AUDIO_VIDEO_DEVICE_IF_CLASS ||
        pIfrDescriptor->bInterfaceClass == WIRELESS_CONTROLLER_IF_CLASS ||
        pIfrDescriptor->bInterfaceClass == ACTIVE_SYNC_IF_CLASS ||
        pIfrDescriptor->bInterfaceClass == APPLICATION_SPECIFIC_IF_CLASS ||
        pIfrDescriptor->bInterfaceClass == VENDOR_SPECIFIC_IF_CLASS)
    {
      return true;
    }

    // Allow HID interfaces if they are not for keyboard, or mouse.
    if(pIfrDescriptor->bInterfaceClass == HID_IF_CLASS &&
        (pIfrDescriptor->bInterfaceSubClass != HID_KB_SUBCLASS ||
         pIfrDescriptor->bInterfaceProtocol != HID_KB_PROTOCOL) &&
        (pIfrDescriptor->bInterfaceSubClass != HID_MOUSE_SUBCLASS ||
         pIfrDescriptor->bInterfaceProtocol != HID_MOUSE_PROTOCOL))
    {
      return true;
    }

    // Allow SCSI Mass Storage interfaces
    if(pIfrDescriptor->bInterfaceClass == MASS_STORAGE_IF_CLASS ||
        pIfrDescriptor->bInterfaceSubClass == MASS_STORAGE_SCSI_IF_SUBCLASS)
    {
      return true;
    }
#endif // VNC_NO_LIBUSB
    // Reject everything else
    return false;
  }

  /// \brief Checks if a USB device is at all of interest.
  ///
  /// If the device is not considered to be useful in any way to discovery,
  /// then it will be not passed to the Discovery SDK.
  ///
  /// \param pDevice The device to check
  ///
  /// \return true if the device should be passed to the Discovery SDK, false
  /// otherwise.
  static bool isDeviceOfInterest(VNCDeviceInfo* pDevice)
  {
    // Only consider devices which have at least one interface (of interest, as
    // defined by isInterfaceOfInterest).
    if(!pDevice || pDevice->interfacesCount == 0)
    {
      return false;
    }
    // Look for at least one interface, either directly under device, or under
    // an available configuration.
    for(size_t i = 0; i < pDevice->interfacesCount; ++i)
    {
      if(pDevice->ppInterfaces[i]->type ==
          UsbDeviceProviderInterfaceTypeConfiguration)
      {
        // A configuration that is not available is not of interest.
        VNCDeviceInterface* pConfig = pDevice->ppInterfaces[i];
        if(!pConfig->available)
          continue;
        if(pConfig->subInterfacesCount != 0)
        {
          return true;
        }
      }
      else if(pDevice->ppInterfaces[i]->type ==
          UsbDeviceProviderInterfaceTypeInterface)
      {
        return true;
      }
    }
    return false;
  }

#ifdef _MSC_VER
  // These are wrappers for the libusb_exit and libusb_unref_device. They can
  // be used with AutoCustomDestroy when building Windows binaries. Using the
  // functions directly is not possible, because, being exported functions,
  // they have a different signature.
  void libusb_exit_wrap(libusb_context* pLibusbCtx)
  {
#ifndef VNC_NO_LIBUSB
    libusb_exit(pLibusbCtx);
#endif
  }

  void libusb_unref_device_wrap(libusb_device* pDev)
  {
#ifndef VNC_NO_LIBUSB
    libusb_unref_device(pDev);
#endif
  }
#endif
}

using namespace vncdiscovery;
using namespace vncdeviceprovider_usb;

UsbDeviceProvider::UsbDeviceProvider(VNCDeviceProviderInterface *pInterface, 
          size_t interfaceSize,
          VNCDeviceProviderCallbacks *pCallbacks,
          size_t callbacksSize)
  : VNCDeviceProviderBase(pInterface, interfaceSize,
      pCallbacks, callbacksSize),
    mUsbDeviceInfoProvider(*this, *this)
{ }

UsbDeviceProvider::~UsbDeviceProvider()
{
  for (AddedDevices::iterator it = mAddedDevices.begin();
    it != mAddedDevices.end(); ++it)
  {
    assert(*it);
    if(!(*it)) continue;
    freeDevice(*it);
  }

  for (KnownDevices::iterator it = mKnownDevices.begin();
    it != mKnownDevices.end(); ++it)
  {
    if (it->owned)
    {
      freeDevice(it->device);
    }
    else
    {
      // The device is a deep copy of what was created by the SDK (see
      // createDeviceDeepCopy()). We must call delete on it.
      delete it->device;
    }
  }
}

VNCDiscoverySDKError UsbDeviceProvider::setProperty(const char* pProperty,
    const char* pValue)
{
  LOG_FLOW(this, "UsbDeviceProvider::setProperty");
  if(!pProperty)
  {
    LOG_ERROR(this, "The property is NULL");
    return VNCDiscoverySDKErrorInvalidParameter;
  }
  // Note: pValue can be NULL (it just means unset)
  VNCDiscoverySDKError ret = VNCDiscoverySDKErrorNotSupported;

  if(ret == VNCDiscoverySDKErrorNotSupported)
  {
    ret = mUsbDeviceInfoProvider.setProperty(pProperty, pValue);
  }
  return ret;
}

VNCDiscoverySDKError UsbDeviceProvider::getProperty(const char *pProperty,
    char **ppValue)
{
  LOG_FLOW(this, "UsbDeviceProvider::getProperty");
  if(!pProperty)
  {
    LOG_ERROR(this, "The property is NULL");
    return VNCDiscoverySDKErrorInvalidParameter;
  }
  if(!ppValue)
  {
    LOG_ERROR(this, "The value location is NULL");
    return VNCDiscoverySDKErrorInvalidParameter;
  }
  VNCDiscoverySDKError ret = VNCDiscoverySDKErrorNotSupported;

  const char* pVal = mUsbDeviceInfoProvider.getProperty(pProperty);
  if(pVal)
  {
    *ppValue = allocString(pVal, -1);
    ret = ((*ppValue == NULL) ? VNCDiscoverySDKErrorOutOfMemory
      : VNCDiscoverySDKErrorNone);
  }
  return ret;
}

VNCDiscoverySDKError UsbDeviceProvider::startMonitoring()
{
  LOG_FLOW(this, "UsbDeviceProvider::startMonitoring");
  if(!mUsbDeviceInfoProvider.startMonitoring())
  {
    return VNCDiscoverySDKErrorUnableToInitialize;
  }

  VNCDiscovererEventHandle *handles = NULL;
  size_t handlesCount = mUsbDeviceInfoProvider.getEventHandles(handles);
  if(handlesCount >= 1) {
    assert(handles != NULL);
    for (size_t i = 0; i < handlesCount; ++i) {
      addEventHandle(&handles[i]);
    }
  }

  // Once started, check if all the existing devices are still valid.
  // If any is not valid it will be marked as removed.
  checkExistingDevices();
  // Also check if devices are connected already (they got connected while the
  // Device Provider is stopped and won't be reported by the monitoring).
  findDevices();
  return VNCDiscoverySDKErrorNone;
}

void UsbDeviceProvider::startReporting()
{
  LOG_FLOW(this, "UsbDeviceProvider::startReporting");
  notifyStarted(VNCDiscoverySDKErrorNone);
  reportChangedDevices();
}

VNCDiscoverySDKError UsbDeviceProvider::stopMonitoring()
{
  LOG_FLOW(this, "UsbDeviceProvider::stopMonitoring");
  VNCDiscovererEventHandle *handles = NULL;
  size_t handlesCount = mUsbDeviceInfoProvider.getEventHandles(handles);
  for (size_t i = 0; i < handlesCount; ++i)
  {
    removeEventHandle(&handles[i]);
  }
  mUsbDeviceInfoProvider.stopMonitoring();
  return VNCDiscoverySDKErrorNone;
}

void UsbDeviceProvider::stopReporting()
{
  LOG_FLOW(this, "UsbDeviceProvider::stopReporting");
  notifyStopped(VNCDiscoverySDKErrorNone);
  // Nothing else to do. The base class will update the state and any device
  // will just not be reported.
}

void UsbDeviceProvider::handleEvent(VNCDiscovererEventHandle* pHandle)
{
  assert(handleInEventHandles(*pHandle));
  mUsbDeviceInfoProvider.handleEvent(pHandle);
}

void UsbDeviceProvider::handleTimeout(VNCDiscovererEventHandle* pHandle)
{
  assert(handleInEventHandles(*pHandle));
  mUsbDeviceInfoProvider.handleTimeout();
  (void)pHandle;
}

#define SET_PROPERTY_ON_DEVICE(name, _key) \
  if(pDevice->propertiesCount > UsbDeviceProviderDeviceProperty##name && \
      pDevice->ppProperties[UsbDeviceProviderDeviceProperty##name]) \
  { \
    char* pKey = allocString((_key), -1); \
    char* pVal = allocString( \
        pDevice->ppProperties[UsbDeviceProviderDeviceProperty##name], -1); \
    if(!pKey || !pVal) \
    { \
      freeString(pKey); \
      freeString(pVal); \
      freeArray(pProperties, size); \
      return 0; \
    } \
    pProperties[size].key = pKey; \
    pProperties[size].value = pVal; \
    ++size; \
  }

size_t UsbDeviceProvider::getDeviceProperties(const VNCDeviceInfo *pDevice,
      VNCDiscoverySDKKeyValPair **ppProperties)
{
  LOG_FLOW(this, "UsbDeviceProvider::getDeviceProperties");
  // Write the bus number, device number, vendor and product IDs and serial
  // number.
  VNCDiscoverySDKKeyValPair *pProperties = allocKeyValPairArray(6);
  if(!pProperties)
  {
    return 0;
  }

  size_t size = 0;

  SET_PROPERTY_ON_DEVICE(BusNum, "BusNum");
  SET_PROPERTY_ON_DEVICE(DevNum, "DevNum");
  SET_PROPERTY_ON_DEVICE(VendorId, "VendorId");
  SET_PROPERTY_ON_DEVICE(ProductId, "ProductId");
  SET_PROPERTY_ON_DEVICE(Serial, "Serial");
  SET_PROPERTY_ON_DEVICE(HasNcm, "HasNcm");

  *ppProperties = pProperties;
  return size;
}

void UsbDeviceProvider::usbDeviceAdded(const UsbDevice& dev)
{
#ifndef VNC_NO_LIBUSB
  LOG_FLOW(this, "UsbDeviceProvider::usbDeviceAdded");
  // Check if the device is already added. If a new device, then store the
  // device ID. Create a device to pass to the SDK.
  vnc_uint32_t id = dev.composeId();
  if(isAddedDeviceDuplicate(id))
  {
    LOG_DEBUG_F(this, "Device with ID %d is added again (syspath='%s'). "
        "Ignoring.", id, dev.sysPath());
    return;
  }

  LOG_DEBUG_F(this, "New device added (bus=%03u, dev=%03u, path='%s')",
      dev.busNum(), dev.devNum(), dev.sysPath());

  libusb_context* pLibusbCtx = NULL;

  // Initialize libusb
  int err = libusb_init(&pLibusbCtx);
  if (err != 0)
  {
    LIBUSB_ERROR("Unable to initialize libusb for the monitor", err);
    return;
  }
  assert(pLibusbCtx);

#ifdef _MSC_VER
  AutoCustomDestroy<libusb_context> libusbCtx(&libusb_exit_wrap, pLibusbCtx);
#else
  AutoCustomDestroy<libusb_context> libusbCtx(&libusb_exit, pLibusbCtx);
#endif
  pLibusbCtx = NULL;

  // Get the (libusb) USB device
  libusb_device* pDev = getUsbDevice(libusbCtx.get(), dev.busNum(), dev.devNum());
  if(pDev)
  {
#ifdef _MSC_VER
    AutoCustomDestroy<libusb_device> device(&libusb_unref_device_wrap, pDev);
#else
    AutoCustomDestroy<libusb_device> device(&libusb_unref_device, pDev);
#endif
    pDev = NULL;

    // Create the device to be passed to the SDK
    VNCDeviceProviderBase::DeviceAutoPtr sdkDevice(*this, 
        createDevice(libusbCtx.get(), device.get(),
          dev.sysPath(), dev.serial()));
    if(sdkDevice)
    {
      sdkDevice->id = id;
      if(getState() == VNCDeviceProviderBase::StateReporting)
      {
        if(isDeviceOfInterest(sdkDevice.get()))
        {
          KnownDevice knownDevice = { createDeviceDeepCopy(sdkDevice.get()), false };
          mKnownDevices.insert(knownDevice);
          notifyDeviceFound(sdkDevice.release());
        }
        else
        {
          KnownDevice knownDevice = { sdkDevice.get(), true };
          mKnownDevices.insert(knownDevice);
          sdkDevice.release();
        }
      }
      else
      {
        // Only monitoring, so save the device for later (when reporting is
        // enabled).
        deviceAddedWhileMonitoring(sdkDevice.get());
        sdkDevice.release();
      }
    }
  }
#else
  LOG_CRITICAL_ERROR(this, "The device provider is built "
      "without libusb functionality, unable to handle new devices.");
#endif // VNC_NO_LIBUSB
}

void UsbDeviceProvider::usbDeviceRemoved(vnc_uint32_t id)
{
  LOG_FLOW(this, "UsbDeviceProvider::usbDeviceRemoved");
  if(getState() == VNCDeviceProviderBase::StateReporting)
  {
    KnownDevices::iterator it = mKnownDevices.begin();
    while (it != mKnownDevices.end())
    {
      if (it->device->id == id)
      {
        break;
      }
      ++it;
    }
    if (it == mKnownDevices.end())
    {
      LOG_WARNING_F(this, "Device with ID %d is removed, but not known.", id);
      return;
    }
    if (it->owned)
    {
      freeDevice(it->device);
    }
    else
    {
      // The device is a deep copy of what was created by the SDK (see
      // createDeviceDeepCopy()). We must call delete on it.
      delete it->device;
    }
    mKnownDevices.erase(it);
    notifyDeviceRemoved(id);
  }
  else
  {
    deviceRemovedWhileMonitoring(id);
  }
}


void UsbDeviceProvider::setTimerForUsbDeviceInfoProvider(
          VNCDiscoverySDKTimeoutMicroseconds usecs)
{
  // Use the first event handle as the timer handle as it is as good as
  // any other.
  VNCDiscovererEventHandle *handles = NULL;
  size_t handlesCount = mUsbDeviceInfoProvider.getEventHandles(handles);
  if (handlesCount < 1 || handles == NULL)
  {
    return;
  }
  setTimer(&handles[0], usecs);
}

#ifdef LOG_DEVICE
#define LOG_PROPERTY(type, name, val) LOG_DEBUG_F(this, \
    "%s property %s: %s", type, name, val)
#else
#define LOG_PROPERTY(type, name, val)
#endif

#define SET_DEVICE_PROPERTY(name, size, fmt, val) do {\
  assert(UsbDeviceProviderDeviceProperty##name < propertiesCount); \
  dev->ppProperties[UsbDeviceProviderDeviceProperty##name] = \
    allocString(NULL, (size)); \
  if(!dev->ppProperties[UsbDeviceProviderDeviceProperty##name]) \
  { \
    return NULL; \
  } \
  snprintf(dev->ppProperties[UsbDeviceProviderDeviceProperty##name], \
    (size), (fmt), (val)); \
  LOG_PROPERTY("device", #name, \
      dev->ppProperties[UsbDeviceProviderDeviceProperty##name]); \
} while(0)

#define SET_DEVICE_PROPERTY_FROM_STRING(name, val) do {\
  assert(UsbDeviceProviderDeviceProperty##name < propertiesCount); \
  dev->ppProperties[UsbDeviceProviderDeviceProperty##name] = \
    allocString(val, -1); \
  if(!dev->ppProperties[UsbDeviceProviderDeviceProperty##name]) \
  { \
    return NULL; \
  } \
  LOG_PROPERTY("device", #name, \
      dev->ppProperties[UsbDeviceProviderDeviceProperty##name]); \
} while(0)

VNCDeviceInfo* UsbDeviceProvider::createDevice(libusb_context* pLibusbCtx,
    libusb_device* pDevice, const char* pSysPath, const char* pSerial)
{
#ifndef VNC_NO_LIBUSB
  // Retrieve the device descriptor
  libusb_device_descriptor devDescriptor;
  int err = libusb_get_device_descriptor(pDevice, &devDescriptor);
  if(err != 0)
  {
    LIBUSB_ERROR("Unable to retrieve the device descriptor", err);
    return NULL;
  }

  size_t propertiesCount = UsbDeviceProviderDevicePropertyHasNcm+1;
  size_t configurationCount = devDescriptor.bNumConfigurations;
  ConfigDescriptors rawConfigDescriptors;
  if (!rawConfigDescriptors.allocate(configurationCount))
  {
    return NULL;
  }

  // Allocate the SDK device
  VNCDeviceProviderBase::DeviceAutoPtr dev(*this,
      allocDevice((vnc_uint32_t)propertiesCount,
        (vnc_uint32_t)configurationCount));
  if(!dev)
  {
    return NULL;
  }
  dev->type = UsbDeviceProviderDeviceTypeUsb;
  dev->pAddress = allocString(pSysPath, -1);
  if(!dev->pAddress)
  {
    return NULL;
  }
  dev->pProviderName = allocString("usb", -1);
  if(!dev->pProviderName)
  {
    return NULL;
  }
  vnc_uint8_t busnum = libusb_get_bus_number(pDevice);
  vnc_uint8_t devnum = libusb_get_device_address(pDevice);

  // busnum
  SET_DEVICE_PROPERTY(BusNum, BUS_DEV_NUM_SIZE+1, "%u", busnum);

  // devnum
  SET_DEVICE_PROPERTY(DevNum, BUS_DEV_NUM_SIZE+1, "%u", devnum);

  // vid
  SET_DEVICE_PROPERTY(VendorId, VID_PID_SIZE+1, "%04x",
      devDescriptor.idVendor);

  // pid
  SET_DEVICE_PROPERTY(ProductId, VID_PID_SIZE+1, "%04x",
      devDescriptor.idProduct);

  // USB device class
  SET_DEVICE_PROPERTY(DeviceClass, USB_CLASS_SIZE+1, "%u",
      devDescriptor.bDeviceClass);

  // USB device sub-class
  SET_DEVICE_PROPERTY(DeviceSubClass, USB_CLASS_SIZE+1, "%u",
      devDescriptor.bDeviceSubClass);

  // USB device protocol
  SET_DEVICE_PROPERTY(DeviceProtocol, USB_CLASS_SIZE+1, "%u",
      devDescriptor.bDeviceProtocol);

  bool needSerial = true;
  if(pSerial)
  {
    // If the serial number is available, use it.
    SET_DEVICE_PROPERTY_FROM_STRING(Serial, pSerial);
    needSerial = false;
  }

  // Retrieve the serial number (a device handle must be obtained for
  // retrieveing the serial number string and configuration descriptors).
  char serial[MAX_SERIAL_SIZE+1];
  serial[0] = '\0';

  libusb_device_handle* pDevHandle = NULL;
  err = libusb_open(pDevice, &pDevHandle);
  if(err != 0)
  {
    LIBUSB_WARNING("Unable to open device handle for retrieving "
                   "serial number and descriptors", err);
    if(pDevHandle)
    {
      libusb_close(pDevHandle);
      pDevHandle = NULL;
    }
  }
  else
  {
    if(pDevHandle)
    {
      err = libusb_get_string_descriptor_ascii(pDevHandle,
                                        devDescriptor.iSerialNumber,
                                        (unsigned char*)serial,
                                        MAX_SERIAL_SIZE+1);
      if(err < 0)
      {
        LIBUSB_WARNING("Unable to retrieve the serial number", err);
      }
      /* Can't use libusb_get_config_descriptor() here as 
       * libusb has a FIXME to handle concatinating multiple
       * unknown descriptors into the extra member. */
      err = getConfigDescriptors(pDevHandle, rawConfigDescriptors);
      if(err < 0)
      {
        LIBUSB_WARNING("Unable to get all config descriptors", err);
      }
      libusb_close(pDevHandle);
      pDevHandle = NULL;
    }
  }
  if (needSerial)
  {
    SET_DEVICE_PROPERTY_FROM_STRING(Serial, serial);
  }

  // Get information about all configurations. Mark the active configuration.
  libusb_config_descriptor* pActiveConfig = NULL;
  err = libusb_get_active_config_descriptor(pDevice, &pActiveConfig);
  if(err != 0)
  {
    LIBUSB_ERROR("Unable to retrieve the active config descriptor", err);
    libusb_free_config_descriptor(pActiveConfig);
    return NULL;
  }
  uint8_t activeConfigVal = pActiveConfig->bConfigurationValue;
  libusb_free_config_descriptor(pActiveConfig);
  pActiveConfig = NULL;

  for(uint8_t i = 0; i < configurationCount; ++i)
  { // set all the configurations
    libusb_config_descriptor* pConfig = NULL;
    err = libusb_get_config_descriptor(pDevice, i, &pConfig);
    if(err != 0)
    {
      LIBUSB_ERROR("Unable to retrieve config descriptor", err);
      libusb_free_config_descriptor(pConfig);
      return NULL;
    }
    VNCDeviceInterface* pInt = createConfiguration(pLibusbCtx, pConfig,
        pSysPath, (activeConfigVal == pConfig->bConfigurationValue),
        rawConfigDescriptors[i], rawConfigDescriptors.length(i));
    libusb_free_config_descriptor(pConfig);
    if(!pInt)
    {
      return NULL;
    }
    dev->ppInterfaces[i] = pInt;
    pInt = NULL;
  }

  bool hasNcm = false;
  for(uint8_t i = 0; i < configurationCount; ++i)
  {
    const VNCDeviceInterface* conf = dev->ppInterfaces[i];
    for(vnc_uint32_t j = 0; j < conf->subInterfacesCount; ++j)
    {
      const VNCDeviceInterface* iface = conf->ppSubInterfaces[j];

      if(iface->type != UsbDeviceProviderInterfaceTypeInterface)
      {
        continue;
      }

      if(iface->propertiesCount <= UsbDeviceProviderInterfaceSubClass)
      {
        continue;
      }

      const char* pClass = iface->ppProperties[UsbDeviceProviderInterfaceClass];
      const char* pSubClass =
        iface->ppProperties[UsbDeviceProviderInterfaceSubClass];

      if(strcmp(pClass, CDC_CONTROL_IF_CLASS_STR) != 0)
      {
        continue;
      }

      if (strcmp(pSubClass, CDC_CONTROL_NCM_SUBCLASS_STR) != 0)
      {
        continue;
      }

      hasNcm = true;
      break;
    }

    if(hasNcm)
    {
      break;
    }
  }

  SET_DEVICE_PROPERTY_FROM_STRING(HasNcm, hasNcm ? "1" : "0");

  return dev.release();
#else
  LOG_CRITICAL_ERROR(this, "The device provider is built "
      "without libusb functionality, unable to create libusb device.");
  return NULL;
#endif // VNC_NO_LIBUSB
}

#define SET_INTERFACE_PROPERTY(name, size, fmt, val) do { \
  assert(UsbDeviceProvider##name < propertiesCount); \
  pInt->ppProperties[UsbDeviceProvider##name] = \
    allocString(NULL, (size)); \
  if(!pInt->ppProperties[UsbDeviceProvider##name]) \
  { \
    freeDeviceInterface(pInt); \
    return NULL; \
  } \
  snprintf(pInt->ppProperties[UsbDeviceProvider##name], \
    (size), (fmt), (val)); \
  LOG_PROPERTY("interface", #name, \
      pInt->ppProperties[UsbDeviceProvider##name]); \
} while(0)

#define SET_INTERFACE_PROPERTY_FROM_STRING(name, val) do {\
  assert(UsbDeviceProvider##name < propertiesCount); \
  pInt->ppProperties[UsbDeviceProvider##name] = \
    allocString(val, -1); \
  if(!pInt->ppProperties[UsbDeviceProvider##name]) \
  { \
    freeDeviceInterface(pInt); \
    return NULL; \
  } \
  LOG_PROPERTY("interface", #name, \
      pInt->ppProperties[UsbDeviceProvider##name]); \
} while(0)

VNCDeviceInterface* UsbDeviceProvider::createConfiguration(libusb_context* pLibusbCtx,
        libusb_config_descriptor* pConfig, const char* pSysPath, bool isActive,
        const unsigned char* configDesc, size_t configDescLength)
{
#ifndef VNC_NO_LIBUSB
  size_t propertiesCount = UsbDeviceProviderConfigurationValue+1;
  size_t interfacesCount = 0;
  // count the interfaces that are of interest.
  for(uint8_t ifrIndex = 0; ifrIndex < pConfig->bNumInterfaces; ++ifrIndex)
  {
    const libusb_interface* pIfr = &((pConfig->interface)[ifrIndex]);
    if(!pIfr)
    {
      continue;
    }

    // retrieve each alternative setting
    for(int altSettingIndex = 0; 
        altSettingIndex < pIfr->num_altsetting; ++altSettingIndex)
    {
      const libusb_interface_descriptor* pIfrDescriptor =
                    &((pIfr->altsetting)[altSettingIndex]);

      // Ignore interfaces that would not be useful for discovery
      if(!isInterfaceOfInterest(pIfrDescriptor))
      {
        continue;
      }
      ++interfacesCount;
    }
  }
  const size_t iadCount = countInterfaceAssociations(configDesc,
      configDesc + configDescLength);

  interfacesCount += iadCount;

  // Set the configuration details and properties
  VNCDeviceInterface* pInt = allocDeviceInterface((vnc_uint32_t)propertiesCount,
      (vnc_uint32_t)interfacesCount);
  pInt->type = UsbDeviceProviderInterfaceTypeConfiguration;
  pInt->available = isActive;
  SET_INTERFACE_PROPERTY(ConfigurationValue, CONFIG_VALUE_SIZE+1,
      "%u", pConfig->bConfigurationValue);

  // add the interfaces that are of interest
  size_t index = 0;
  for(uint8_t ifrIndex = 0; ifrIndex < pConfig->bNumInterfaces; ++ifrIndex)
  {
    assert(index <= interfacesCount);
    const libusb_interface* pIfr = &((pConfig->interface)[ifrIndex]);
    if(!pIfr)
    {
      continue;
    }

    // retrieve each alternative setting
    for(int altSettingIndex = 0; 
        altSettingIndex < pIfr->num_altsetting; ++altSettingIndex)
    {
      const libusb_interface_descriptor* pIfrDescriptor =
                    &((pIfr->altsetting)[altSettingIndex]);

      // Ignore interfaces that would not be useful for discovery
      if(!isInterfaceOfInterest(pIfrDescriptor))
      {
        continue;
      }

      VNCDeviceInterface* pInterface = createInterface(pLibusbCtx,
          pIfrDescriptor, pSysPath, isActive, pConfig->bConfigurationValue);
      if(!pInterface)
      {
        freeDeviceInterface(pInt);
        return NULL;
      }
      assert(index < interfacesCount);
      pInt->ppSubInterfaces[index] = pInterface;
      ++index;
      pInterface = NULL;
    }
  }

  const unsigned char* offset = configDesc;
  // add the interface associations that are of interest
  for(uint8_t iadIndex = 0; iadIndex < iadCount; ++iadIndex)
  {
    VNCDeviceInterface* pIad = createInterfaceAssociation(
        offset, configDesc + configDescLength, isActive);
    if (!pIad)
    {
      freeDeviceInterface(pInt);
      return NULL;
    }
    assert(index <= interfacesCount);
    pInt->ppSubInterfaces[index] = pIad;
    ++index;
    pIad = NULL;
  }

  assert(index == interfacesCount);

  return pInt;
#else
  LOG_CRITICAL_ERROR(this, "The device provider is built "
      "without libusb functionality, unable to create libusb context.");
  return NULL;
#endif // VNC_NO_LIBUSB
}

VNCDeviceInterface* UsbDeviceProvider::createInterface(libusb_context* pLibusbCtx,
    const libusb_interface_descriptor* pIfrDescr,
    const char* pSysPath, bool isActive, vnc_uint8_t configValue)
{
  (void) pLibusbCtx;
#ifndef VNC_NO_LIBUSB
  size_t propertiesCount = UsbDeviceProviderInterfaceDevPath+1;
  VNCDeviceInterface* pInt = allocDeviceInterface(
    (vnc_uint32_t)propertiesCount, 0);
  pInt->type = UsbDeviceProviderInterfaceTypeInterface;
  pInt->available = isActive;

  SET_INTERFACE_PROPERTY(InterfaceClass, USB_CLASS_SIZE+1,
      "%u", pIfrDescr->bInterfaceClass);

  SET_INTERFACE_PROPERTY(InterfaceSubClass, USB_CLASS_SIZE+1,
      "%u", pIfrDescr->bInterfaceSubClass);

  SET_INTERFACE_PROPERTY(InterfaceProtocol, USB_CLASS_SIZE+1,
      "%u", pIfrDescr->bInterfaceProtocol);

  SET_INTERFACE_PROPERTY(InterfaceNumber, USB_CLASS_SIZE+1,
      "%u", pIfrDescr->bInterfaceNumber);

  SET_INTERFACE_PROPERTY(InterfaceAlternateSetting, USB_CLASS_SIZE+1,
      "%u", pIfrDescr->bAlternateSetting);

  if(pIfrDescr->bInterfaceClass == HID_IF_CLASS)
  {
    char devPath[MAX_DEVPATH];
    size_t size = mUsbDeviceInfoProvider.getDevName(pSysPath, configValue,
        pIfrDescr->bInterfaceNumber,
        pIfrDescr->bAlternateSetting,
        UsbDevice::InterfaceTypeHidraw,
        devPath, MAX_DEVPATH);
    if(size > MAX_DEVPATH)
    {
      LOG_WARNING_F(this, "Need more space to store the device path, "
          "needed=%u", size);
    }
    else if(size > 0)
    {
      SET_INTERFACE_PROPERTY_FROM_STRING(InterfaceDevPath, devPath);
    }
  }
  else if(pIfrDescr->bInterfaceClass == CDC_CONTROL_IF_CLASS &&
      pIfrDescr->bInterfaceSubClass == ABSTRACT_MODEM_SUBCLASS &&
      pIfrDescr->bInterfaceProtocol == 0xFF)
  { // S60 serial communications channel. S60 devices have two tty interfaces
    // set up, one is this one that can be used for "custom" serial
    // connections. The other one is for protocol 0x01, used for AT-commands.
    // From the point of view of discovering VNC servers, this is not really
    // needed.
    char devPath[MAX_DEVPATH];
    size_t size = mUsbDeviceInfoProvider.getDevName(pSysPath, configValue,
        pIfrDescr->bInterfaceNumber,
        pIfrDescr->bAlternateSetting,
        UsbDevice::InterfaceTypeTty,
        devPath, MAX_DEVPATH);
    if(size > MAX_DEVPATH)
    {
      LOG_WARNING_F(this, "Need more space to store the device path, "
          "needed=%u", size);
    }
    else if(size > 0)
    {
      SET_INTERFACE_PROPERTY_FROM_STRING(InterfaceDevPath, devPath);
    }
  }

  return pInt;

#else
  LOG_CRITICAL_ERROR(this, "The device provider is built "
      "without libusb functionality, unable to create libusb interface.");
  return NULL;
#endif // VNC_NO_LIBUSB
}

VNCDeviceInterface* UsbDeviceProvider::createInterfaceAssociation(
    const unsigned char*& start, const unsigned char* end, bool isActive)
{
  const unsigned char* desc = NULL;
  vnc_uint8_t bLength = 0;
  vnc_uint8_t bDescriptorType = 0;
  // Find the next interface association descriptor in desc
  do
  {
    if (!getNextDescriptor(start, end, desc, bLength, bDescriptorType))
    {
      return NULL;
    }
  } while (bDescriptorType != INTERFACE_ASSOCIATION_TYPE ||
      bLength < INTERFACE_ASSOCIATION_MIN_LENGTH);

  vnc_uint8_t bFirstInterface = desc[2];
  vnc_uint8_t bInterfaceCount = desc[3];
  vnc_uint8_t bFunctionClass = desc[4];
  vnc_uint8_t bFunctionSubClass = desc[5];
  vnc_uint8_t bFunctionProtocol = desc[6];

  size_t propertiesCount = UsbDeviceProviderInterfaceAssociationCount+1;
  VNCDeviceInterface* pInt = allocDeviceInterface(
    (vnc_uint32_t)propertiesCount, 0);
  pInt->type = UsbDeviceProviderInterfaceTypeInterfaceAssociation;
  pInt->available = isActive;

  SET_INTERFACE_PROPERTY(InterfaceAssociationClass, USB_CLASS_SIZE+1,
      "%u", bFunctionClass);

  SET_INTERFACE_PROPERTY(InterfaceAssociationSubClass, USB_CLASS_SIZE+1,
      "%u", bFunctionSubClass);

  SET_INTERFACE_PROPERTY(InterfaceAssociationProtocol, USB_CLASS_SIZE+1,
      "%u", bFunctionProtocol);

  SET_INTERFACE_PROPERTY(InterfaceAssociationFirstInterface, USB_CLASS_SIZE+1,
      "%u", bFirstInterface);

  SET_INTERFACE_PROPERTY(InterfaceAssociationCount, USB_CLASS_SIZE+1,
      "%u", bInterfaceCount);

  return pInt;
}

void UsbDeviceProvider::findDevices()
{
  LOG_FLOW(this, "UsbDeviceProvider::findDevices");
  // Go through the list of existing devices and see if any of them
  // can be an entity (that is not discovered yet).
  if(!mUsbDeviceInfoProvider.startEnumeration())
  {
    return;
  }
  UsbDevice* pDev = NULL;
  while(NULL != (pDev = mUsbDeviceInfoProvider.nextDevice()))
  {
    usbDeviceAdded(*pDev);
    delete pDev;
  }
  mUsbDeviceInfoProvider.stopEnumeration();
}

void UsbDeviceProvider::checkExistingDevices()
{
#ifndef VNC_NO_LIBUSB
  LOG_FLOW(this, "UsbDeviceProvider::checkExistingDevices");

  libusb_context* pLibusbCtx = NULL;
  // Initialize libusb
  int err = libusb_init(&pLibusbCtx);
  if (err != 0)
  {
    LIBUSB_ERROR("Unable to initialize libusb for the monitor", err);
    return;
  }
  assert(pLibusbCtx);

  // Go through the known devices and see if they can still retrieved from
  // libusb.
  KnownDevices::iterator it = mKnownDevices.begin();
  while(it != mKnownDevices.end())
  {
    vnc_uint8_t bus, dev;
    VNCDeviceInfo* pKnownDev = it->device;
    ++it;
    UsbDevice::getBusAndDevNums(pKnownDev->id, bus, dev);
    libusb_device* pDev = getUsbDevice(pLibusbCtx, bus, dev);
    if(!pDev)
    {
      // Device has been removed.
      usbDeviceRemoved(pKnownDev->id);
    }
    else
    {
      char newVid[5];
      char newPid[5];
      libusb_device_descriptor devDescriptor;
      libusb_get_device_descriptor(pDev, &devDescriptor);
      snprintf(newVid, sizeof(newVid), "%04x", devDescriptor.idVendor);
      snprintf(newPid, sizeof(newPid), "%04x", devDescriptor.idProduct);

      const char* const oldVid = pKnownDev->ppProperties[UsbDeviceProviderDevicePropertyVendorId];
      const char* const oldPid = pKnownDev->ppProperties[UsbDeviceProviderDevicePropertyProductId];

      if (strcmp(newVid, oldVid) != 0 || strcmp(newPid, oldPid) !=0)
      {
        // The VIDs and/or PIDs are different, assume it's a different device
        usbDeviceRemoved(pKnownDev->id);
      }
    }
    libusb_unref_device(pDev);
  }

  libusb_exit(pLibusbCtx);
  pLibusbCtx = NULL;
#else
  LOG_CRITICAL_ERROR(this, "The device provider is built "
      "without libusb functionality, unable to check existing devices.");
#endif // VNC_NO_LIBUSB
}

bool UsbDeviceProvider::handleInEventHandles(VNCDiscovererEventHandle handle)
{
  VNCDiscovererEventHandle *handles = NULL;
  size_t handlesCount = mUsbDeviceInfoProvider.getEventHandles(handles);
  if (handlesCount < 1)
  {
    return false;
  }
  assert(handles != NULL);
  for (size_t i = 0; i < handlesCount; ++i)
  {
    if (handles[i] == handle)
    {
      return true;
    }
  }
  return false;
}

VNCDeviceInfo* UsbDeviceProvider::createDeviceDeepCopy(VNCDeviceInfo* device)
{
  VNCDeviceInfo* pCopy = new VNCDeviceInfo;
  memset(pCopy, 0, sizeof(VNCDeviceInfo));
  memcpy(pCopy, device, sizeof(VNCDeviceInfo));

  return pCopy;
}

void UsbDeviceProvider::deviceAddedWhileMonitoring(VNCDeviceInfo* device)
{
  // A device should not be added twice.
  assert(mAddedDevices.end() == mAddedDevices.find(device));
  assert(device);
  if(!device)
  {
    return;
  }
  mAddedDevices.insert(device);
}

void UsbDeviceProvider::deviceRemovedWhileMonitoring(vnc_uint32_t id)
{
  // First check that the removed device is in the list of devices added while
  // monitoring. If it is, remove that one. If not, then check it's in the
  // known list. If it can be found there, then mark it for removal (when
  // report is started again).
  for(AddedDevices::iterator it = mAddedDevices.begin();
      it != mAddedDevices.end(); ++it)
  {
    assert(*it);
    if(!(*it)) continue;
    if((*it)->id == id)
    {
      freeDevice(*it);
      mAddedDevices.erase(it);
      return;
    }
  }
  for(KnownDevices::iterator it = mKnownDevices.begin();
      it != mKnownDevices.end(); ++it)
  {
    if (it->device->id == id)
    {
      assert(mRemovedDevices.end() == mRemovedDevices.find(id));
      mRemovedDevices.insert(id);
      return;
    }
  }
  LOG_WARNING_F(this, "Device with ID %d is removed, but not known.", id);
}

void UsbDeviceProvider::reportChangedDevices()
{
  // First report removed devices, emptying the removed device list.
  assert(getState() == StateReporting);
  RemovedDevices::iterator removed = mRemovedDevices.begin();
  while(removed != mRemovedDevices.end())
  {
    usbDeviceRemoved(*removed);
    mRemovedDevices.erase(removed++);
  }
  // Then report the devices that have been added, emptying that list. As the
  // removed devices were handled first there shouldn't be any duplicates.
  AddedDevices::iterator added = mAddedDevices.begin();
  while(added != mAddedDevices.end())
  {
    assert(*added);
    VNCDeviceProviderBase::DeviceAutoPtr sdkDevice(*this, *added);
    mAddedDevices.erase(added++);
    if(!sdkDevice)
    {
      continue;
    }
    assert(!isAddedDeviceDuplicate(sdkDevice->id));
    if(isDeviceOfInterest(sdkDevice.get()))
    {
      KnownDevice knownDevice = { createDeviceDeepCopy(sdkDevice.get()), false };
      mKnownDevices.insert(knownDevice);
      notifyDeviceFound(sdkDevice.release());
    }
    else
    {
      KnownDevice knownDevice = { sdkDevice.get(), true };
      mKnownDevices.insert(knownDevice);
      sdkDevice.release();
    }
  }
  assert(mRemovedDevices.empty());
  assert(mAddedDevices.empty());
}

bool UsbDeviceProvider::isAddedDeviceDuplicate(vnc_uint32_t id) const
{
  // First look through the added devices to see if it is a duplicate. If
  // already reporting, the added set will be empty.
  for(AddedDevices::const_iterator it = mAddedDevices.begin();
      it != mAddedDevices.end(); ++it)
  {
    if(id == (*it)->id)
    {
      // Already an add pending, so it's a duplicate.
      return true;
    }
  }

  // Then look through the removed devices. If the device is in that set, then
  // it means that this new one is not a duplicate, but just a devices added
  // with the same ID as the previously removed one.
  for(RemovedDevices::const_iterator it = mRemovedDevices.begin();
      it != mRemovedDevices.end(); ++it)
  {
    if(id == *it)
    {
      // No add pending, but a remove pending (which means it would be in the
      // known list), so this is a new device added with the same ID as a
      // previous one.
      return false;
    }
  }

  // Finally, check if the device is in the list of Known Devices.
  for(KnownDevices::const_iterator it = mKnownDevices.begin();
      it != mKnownDevices.end(); ++it)
  {
    if (it->device->id == id)
    {
      // This is a duplicate, as it wasn't yet marked for removal
      return true;
    }
  }
  return false;
}

extern "C" VNCDeviceProvider *VNCCALL VNCDeviceProviderInitialize(
    VNCDeviceProviderInterface *pInterface, 
    size_t interfaceSize,
    VNCDeviceProviderCallbacks *pCallbacks,
    size_t callbacksSize)
{
  return new (std::nothrow) UsbDeviceProvider(pInterface, interfaceSize, 
                    pCallbacks, callbacksSize);
}

